;*******************************************************************************
;
;	Filename:		BarIX.asm
;	Date:				21 Jan 20156
;	File Version:	1.0
;	Author:			Larry Cicchinelli
;	Description:
;		This program uses a PIC 12F1572 to (somewhat) emulate a 555 timer.
;
;		PIC12F1572 I/O:
;			pin 1: VDD
;			pin 2: RA5							Pulse output
;			pin 3: RA4, AN3					Mode divider
;			pin 4: RA3, ~MCLR					Trigger
;			pin 5: RA2, AN2					Pulse Width potentiometer
;			pin 6: RA1, AN1, ICSPCLK		Delay potentiometer
;			pin 7: RA0, AN0, ISCPDAT		Range divider
;			pin 8: VSS
;
;		PIC12F1572 I/O - DEBUG!!
;			pin 1: VDD
;			pin 2: RA5							Pulse output
;			pin 3: RA4, AN3					Trigger
;			pin 4: ~MCLR
;			pin 5: RA2, AN2					Pulse Width potentiometer
;			pin 6: ICSPCLK
;			pin 7: ISCPDAT
;			pin 8: VSS
;
; IMPORTANT
;	Both Mode and Range are read only once - when the program starts.
;	Both the Pulse Width and Pulse Delay are read immdediately preceding the
;		generation of each output occurance.
;	Response time can be improved by reading the Pulse Width and Pulse Delay
;		values only one time.  See the code which executes if FAST is defined.
;	If a function changes the BSR, it MUST set it back to PORTA before returning.
; NOTE: the T1 registers are in the same bank (0) as PORTA
;
; Definitions used by the program which are associated with specific I/O pins:
;	AV_PW			- analog voltage which defines the output pulse width
;	AV_DELAY		- analog voltage which defines the delay preceding the output pulse
;						as well as define the "0" time of the astable multivibrator
;	AV_MODE		- analog voltage defining operating mode - most sig 3 bits
;	AV_RANGE		- analog voltage defining pulse width range/delay - most sig 3 bits
;	O_PULSE		- output pulse pin
;	I_TRIGGER	- input trigger pin
;
;	Programming notes:
;		The CPU clock is the internal 8MHz oscillator: (see Get.Range)
;			Ranges 0 - 4 =- PLL on: Fosc = 32MHz
;			Ranges 5 - 7 = PLL off: Fosc = 8MHz
;			The A/D sample clock is always 1MHz
;		All functions that change the BSR return with BSR = 0
;		The Mode and Range/Resolution values are read only once - at startup
;
;	Timer 1 is used as a programmable timer for Ranges 1-7:
;		source = Fosc/4, prescale = 8: T1 clock is always Fcpu/32
;		Range 0 does not use the timer
;		
;							T1			T1			Software
;		Resolution		Clk		Count		Divisor		Max PW
; PLL on: Fcpu = 8MHz*4 = 32MHz
;		0 - 1us										A/D		1ms
;		1 - 10us			1Mhz		10				A/D		10ms
;		2 - 100us		1Mhz		100			A/D		100ms
;		3 - 1ms			1Mhz		1000			A/D		1s
;		4 - 10ms			1Mhz		10000			A/D		10s
; PLL off: Fcpu = 8MHz
;		5 - 100ms		250KHz	25000			A/D		100s
; PLL off: Fcpu = 2MHz
;		6 - 1s			62.5KHz	62500			A/D		1000s (16.6 mins)
;		7 - 10s			62.5KHz	62500			A/D*10	10,000s (166 mins)
;
;	Modes:
;		0 = One Shot
;		1 = Delayed One Shot
;		2 = Retriggerable One Shot
;		3 = Delayed Retriggerable One Shot
;		4 = Multivibrator
;		5 = Gated Multivibrator
;		6,7 = force to mode = 0
;
;*******************************************************************************
DEBUG		equ	0				; 0 = no debugging
FAST		equ	0				; 1 = bypass some code to speed up the response time
									;		by reading the PW and Delay values once only.
;*******************************************************************************

		#include p12F1572.inc

#if DEBUG == 0	;========= hardware definitions when =NOT= debugging ============
;																	 =====
; CONFIG1
; __config 0xF9E4
 __CONFIG _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_OFF & _MCLRE_OFF & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF
; CONFIG2
; __config 0xFEFF
 __CONFIG _CONFIG2, _WRT_OFF & _PLLEN_OFF & _STVREN_ON & _BORV_LO & _LPBOREN_OFF & _LVP_OFF

AV_RANGE		equ	RA0
AV_DELAY		equ	RA1
AV_PW			equ	RA2
I_TRIGGER	equ	RA3
AV_MODE		equ	RA4
O_PULSE		equ	RA5

TRISA_VAL	equ	b'11011111'
ANSEL_VAL	equ	b'00010111'
ADCH_RANGE	equ	( 0<<CHS0 ) | (1<< ADON )
ADCH_DELAY	equ	( 1<<CHS0 ) | (1<< ADON )
ADCH_PW		equ	( 2<<CHS0 ) | (1<< ADON )
ADCH_MODE	equ	( 3<<CHS0 ) | (1<< ADON )
		
#else ;====================== hardware definitions when debugging ==============

; CONFIG1
; __config 0xF9E4
 __CONFIG _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_OFF & _MCLRE_ON & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF
; CONFIG2
; __config 0xFEFF
 __CONFIG _CONFIG2, _WRT_OFF & _PLLEN_OFF & _STVREN_ON & _BORV_LO & _LPBOREN_OFF & _LVP_ON

AV_PW			equ	RA2
I_TRIGGER	equ	RA4
O_PULSE		equ	RA5

; the following 3 values are modified as necessary to test the software
AV_DELAY_VALUE	equ .0
AV_RANGE_VALUE	equ .0
AV_MODE_VALUE	equ .4
	

TRISA_VAL	equ	b'11011111'
ANSEL_VAL	equ	b'00000100'
ADCH_PW		equ	( 2<<CHS0 )| (1<< ADON )

#endif ;========================================================================

;*******************************************************************************
AD_NO_BITS_0_1	equ	0		; 1 = clear A/D bits 0 & 1 for PW and Delay.
									;		This is done to make it easier to adjust the
									;		potentiometers for the full range of values.
									;		It also forces the resolution to be 4 times the
									;		listed value; ie, 4us, 40us, etc
AD_USE_8_BITS equ		0		; 1 = use upper 8 bits for PW and delay.  This is
									;		done to make it easier to adjust the
									;		potentiometers leaving the resolution unchanged.
									;		However, the full range of values is decreased
									;		by a factor of 4; ie, 250us, 2.5ms, etc
; both AD_NO_BITS_0_1 and AD_USE_8_BITS may be enabled simultaneously, this will
; result essentially in a 6 bit A/D reading.  It will force both the resolution
; and full range to be reduced.

EDGE_REG equ	IOCAP			; IOCAP = trigger on positive edge of input
									; IOCAN = trigger on negative edge of input
;*******************************************************************************

; Common RAM allocations: 16 bytes
Access_RAM		udata_shr	0x70		; 16 bytes starting at address 0x70
ADvalue			res	2
Mode				res	1
Range				res	1

PW_Counter		res	2
Delay_Counter	res	2
T1_Preset		res	2

; Since the total number of bytes of RAM required are 14, the program does not
; need to make use of banked RAM

;Bank0RAM			udata			0x20
temp0				res	2
RangeTemp		res	1
Flags				res	1
FLAGS_RETRIGGER	equ	0

;*******************************************************************************
; Vectors
;*******************************************************************************

	org	0
	goto	Main

	org	0x04								; interrupt vector
ISR.Exit
	retfie


;*******************************************************************************
; MAIN PROGRAM
;*******************************************************************************

MAIN_PROG CODE                      ; let linker place main program

Main
	call		init							; initialize the system hardware
	clrf		Flags

#if AD_USE_8_BITS == 1
	BANKSEL	ADCON1
	bcf		ADCON1, ADFM				; left justify the data
	BANKSEL	PORTA
#endif

	call		Get.Range					; value saved in Range
	call		Get.Mode						; value in W and saved in Mode
; FSR0 is used to point to Common RAM so it upper byte is ALWAYS 0
	clrf		FSR0H							; clear upper byte of pointer

; determine Mode of operation by successive decrementing W
	andlw		0xFF							; Z=1 iff W=0
	btfsc		STATUS, Z					; skip next if Mode <> 0
	goto		One.Shot

	addlw		-1								; W = Range - 1
	btfsc		STATUS, Z					; skip next if Mode <> 1
	goto		Delay.One.Shot

	addlw		-1								; W = Range - 2
	btfsc		STATUS, Z					; skip next if Mode <> 2
	goto		Retrig.One.Shot

	addlw		-1								; W = Range - 3
	btfsc		STATUS, Z					; skip next if Mode <> 3
	goto		Delay.Retrig.One.Shot

	addlw		-1								; W = Range - 4
	btfsc		STATUS, Z					; skip next if Mode <> 4
	goto		Multivibrator

	addlw		-1								; W = Range - 5
	btfsc		STATUS, Z					; skip next if Mode <> 5
	goto		Gated.Multivibrator

Main.100
	clrf		Mode							; force Mode = 0
	goto		One.Shot


; ==============================================================================
; ================================== One Shot ==================================
; ==============================================================================

One.Shot										; mode 0 = one-shot
	call		Get.PW
	movlw		LOW PW_Counter				; get address of value for software counter
	movwf		FSR0L							;		used by Run.T1

One.Shot.10
; wait for trigger
	BANKSEL	EDGE_REG
	bsf		EDGE_REG, I_TRIGGER		; enable + edge detect
One.Shot.30									; wait for trigger
	btfss		IOCAF, I_TRIGGER			; skip next if + edge detected
	goto		One.Shot.30
	btfss		Flags, FLAGS_RETRIGGER	; skip next if retrigger needed
	bcf		EDGE_REG, I_TRIGGER		; disable + edge detect
	bcf		IOCAF, I_TRIGGER			; clear the edge detect flag
	BANKSEL	PORTA
	bsf		PORTA, O_PULSE				; turn on output pulse
	call		Run.T1						; execute pulse width
	bcf		PORTA, O_PULSE				; turn off output pulse

#if FAST == 0
	goto		One.Shot
#else
	goto		One.Shot.10					; do not read A/D again
#endif
; One.Shot

; ==============================================================================
; ============================== Delayed One Shot ==============================
; ==============================================================================

Delay.One.Shot								; mode 1 = delayed one-shot
	call		Get.PW
	call		Get.Delay
Delay.One.Shot.10
; set up for delay
	movlw		LOW Delay_Counter			; get address of value for software counter
	movwf		FSR0L							;		used by Run.T1
; wait for trigger
	BANKSEL	EDGE_REG
	bsf		EDGE_REG, I_TRIGGER		; enable + edge detect
Delay.One.Shot.30							; wait for trigger
	btfss		IOCAF, I_TRIGGER			; skip next if + edge detected
	goto		Delay.One.Shot.30
	btfss		Flags, FLAGS_RETRIGGER	; skip next if retrigger needed
	bcf		EDGE_REG, I_TRIGGER		; disable + edge detect
	bcf		IOCAF, I_TRIGGER			; clear the edge detect flag
	BANKSEL	PORTA
	call		Run.T1						; execute delay time
; set up for PW
	movlw		LOW PW_Counter				; point to the PW counter value
	movwf		FSR0L							;		used by Run.T1
	bsf		PORTA, O_PULSE				; turn on output pulse
	call		Run.T1						; execute pulse width
	bcf		PORTA, O_PULSE				; turn off output pulse

#if FAST == 0
	goto		Delay.One.Shot
#else
	goto		Delay.One.Shot.10			; do not read A/D again
#endif
; Delay.One.Shot

; ==============================================================================
; ============================ Retriggered One Shot ============================
; ==============================================================================

Retrig.One.Shot							; mode 2 = retriggerable one-shot
	bsf		Flags, FLAGS_RETRIGGER
	goto		One.Shot

; ==============================================================================
; ======================= Delayed & Retriggered One Shot =======================
; ==============================================================================

Delay.Retrig.One.Shot					; mode 3 = delayed, retriggerable one-shot

	bsf		Flags, FLAGS_RETRIGGER
	goto		Delay.One.Shot


; ==============================================================================
; =============================== Multivibrator ================================
; ==============================================================================

Multivibrator
#if FAST == 1								; get values one time
	call		Get.Delay
	call		Get.PW
#endif
Multivibrator.10
	call		Do.Period
	goto		Multivibrator.10
; Multivibrator

Do.Period
; set up timer for PW
; the output pulse is turned on "early" in order to enable a square wave closer
; to 50% when the Pulse Width and Off-Time inputs are tied together.
#if FAST == 0
	call		Get.PW
#endif
	bsf		PORTA, O_PULSE				; turn on output pulse
	movlw		LOW PW_Counter				; point to the PW counter value
	movwf		FSR0L							;		used by Run.T1
	call		Run.T1						; execute pulse width
	bcf		PORTA, O_PULSE				; turn off output pulse
; set up timer for off time
#if FAST == 0
	call		Get.Delay
#endif
	movlw		LOW Delay_Counter			; get address of value for software counter
	movwf		FSR0L							;		used by Run.T1
	goto		Run.T1						; execute off time and return
; Do.Period


; ==============================================================================
; ============================ Gated Multivibrator =============================
; ==============================================================================

Gated.Multivibrator
#if FAST == 1								; get values one time
	call		Get.Delay
	call		Get.PW
#endif
Gated.Multivibrator.10
	btfsc		PORTA, I_TRIGGER			; skip next if the gate is off
	call		Do.Period
	goto		Gated.Multivibrator.10
; Gated.Multivibrator


;===============================================================================
;									Main Utility Functions
;===============================================================================

; Start Timer 1 and wait for it to complete
; Enter with:
;		T1_Preset value set
;		FSR0L = address of low byte of x_Counter value
;		FSR1 = used as 16 bit counter - not a pointer
; I_TRIGGER bit indicates retrigger requested

Run.T1
; check for Range = 0
	movfw		Range							; get the range
	andlw		0xFF							; w = 0 iff Range = 0
	btfsc		STATUS, Z					; skip if w <> 0
	goto		Run.Range0					; goto if Range = 0
;
; Ranges 1-7
;
Run.T1.10
; get software counter value
	moviw		FSR0++						; low byte and point to high byte
	movwf		FSR1L
	moviw		FSR0--						; high byte and point back to low byte
	movwf		FSR1H
;	moviw		FSR1--						; adjust for non-0 reading of grounded input
Run.T1.20
	bcf		T1CON, TMR1ON				; turn timer off
	bcf		PIR1, TMR1IF				; reset the overflow flag
; check if done software counter
	moviw		FSR1--						; update software counter
	btfsc		FSR1H, 7						; will be negative if decremented to -1
	goto		Run.T1.40					; goto if done
; set Timer 1 preset and turn it on
	movf		T1_Preset, w				; get low byte of Timer preset count
	movwf		TMR1L
	movf		T1_Preset+1, w				; get high byte
	movwf		TMR1H
	bsf		T1CON, TMR1ON				; turn timer on
	nop										; 3 cycles to make 20 total
	nop										;		seeT1PresetTable
	nop
; wait for timeout
Run.T1.30									; wait for timer to time out
	btfss		PIR1, TMR1IF				; skip if overflow (=timeout)
	goto		Run.T1.30
	goto		Run.T1.20

Run.T1.40
; test for retrigger
	BANKSEL	EDGE_REG
	btfss		IOCAF, I_TRIGGER			; skip next if + edge detected
	goto		Run.T1.50					; exit if no retrigger detected
	bcf		IOCAF, I_TRIGGER			; clear the edge detect flag
	BANKSEL	PORTA
	goto		Run.T1.10					; start timing over
Run.T1.50
	BANKSEL	PORTA
Run.T1.90
	return
; Run.T1
	

; these two lines of code are here in order to remove the necessity for a goto
; and are executed when a retrigger is detected.
Run.Range0.30
	bcf		IOCAF, I_TRIGGER			; clear the edge detect flag
	BANKSEL	PORTA

Run.Range0									; Range = 0 only!
; This functions does not use Timer 1 - it uses instruction execution only!!
; Enter with:
;		FSR0L = address of low byte of x_Counter value
; get the count value into FSR1 used as 16 bit counter - not a pointer
	moviw		FSR0++						; low byte
	movwf		FSR1L
	moviw		FSR0--						; high byte (and point back to low byte)
	movwf		FSR1H
; values of 0-3 will all yield the same PW: about 3usec
	moviw		FSR1--						; adjust
	moviw		FSR1--						;		for fixed
	moviw		FSR1--						;				delays
	
; the following loop executes in exactly 8 instruction cycles = 1us
Run.Range0.10
	; test for x_Counter == 0
	moviw		FSR1--						; update the count value
	btfsc		FSR1H, 7						; skip if new count >= 0
	goto		Run.Range0.20				; exit loop when counter < 0
	nop										; needed to make 8 instruction cycles
	nop
	nop
	goto		Run.Range0.10

Run.Range0.20
; test for retrigger
	BANKSEL	EDGE_REG
	btfsc		IOCAF, I_TRIGGER			; skip next if + edge NOT detected
	goto		Run.Range0.30				; goto if retrigger detected
	BANKSEL	PORTA
	return
; Run.T1.Range0


;===============================================================================
;									Main Operating Functions
;===============================================================================

Get.PW
; Read the A/D
; determine if the value needs to be multiplied by 10 (range 7)
; store value into PW_Counter
	movlw		ADCH_PW						; set up to
	call		AD.Read						;		get Pulse width
; if Mode = 7 multiply ADvalue by 10
	movlw		7
	subwf		Range, w						; Z = 1 if Range = 7
	btfsc		STATUS, Z					; skip next if range <> 7
	call		AD.X10
;
	movfw		ADvalue						; need to save the pulse width
	movwf		PW_Counter					;		so it can be used
	movfw		ADvalue+1					;			repeatitively
	movwf		PW_Counter+1
	return
; Get.PW


Get.Delay
; Read the A/D
; determine if the value needs to be multiplied by 10 (ranges 1 & 7)
; store value into Delay_Counter
#if DEBUG == 0						; if normal operation
	movlw		ADCH_DELAY					; set up to
	call		AD.Read						;		get delay
#else									; if debugging
	movlw		LOW AV_DELAY_VALUE
	movwf		ADvalue
	movlw		HIGH AV_DELAY_VALUE
	movwf		ADvalue+1
#endif
; if Mode = 7 multiply ADvalue by 10
	movlw		7
	subwf		Range, w						; Z = 1 if Range = 7
	btfsc		STATUS, Z					; skip next if range <> 7
	call		AD.X10
;
	movfw		ADvalue						; need to save the pulse width
	movwf		Delay_Counter				;		so it can be used
	movfw		ADvalue+1					;			repeatitively
	movwf		Delay_Counter+1
	return
; Get.Delay


Get.Mode
; exit with the mode in W and Mode
#if DEBUG == 0						; if normal operation
	movlw		ADCH_MODE					; get the Mode channel
	call		Get.Range.Mode				; read the A/D
#else									; if debugging
	movlw		AV_MODE_VALUE				; get the debug mode
#endif ; DEBUG
	movwf		Mode							; store it
	return
; Get.Mode


Get.Range
; exit with
;	Range set using bits 7-9 of A/D reading
;	T1_Preset initialized from the table
;	OSCCON set based on Range
; uses
;	INDF1
#if DEBUG == 0						; if normal operation
	movlw		ADCH_RANGE					; get the Range channel
	call		Get.Range.Mode				; read the A/D
#else									; if debugging
	movlw		AV_RANGE_VALUE				; get the debug mode
#endif ; DEBUG
	movwf		Range							; store it

; Get Timer 1 preset value from T1PresetTable
; get address of the T1 Preset Table into FSR1
	movlw		LOW T1PresetTable
	movwf		FSR1L
	movlw		HIGH T1PresetTable
	movwf		FSR1H
; calculate offset into table
	movfw		Range							; get range value
	lslf		WREG, w						; make table offset value
; calculate table entry address
	addwf		FSR1L, f
	clrf		WREG
	addwfc	FSR1H, f
	bsf		FSR1H, 7						; tell INDF1 to access program memory
; get and save the table value
	moviw		INDF1++						; get the low byte
	movwf		T1_Preset					; save it
	moviw		INDF1++						; get the high byte
	movwf		T1_Preset+1					; save it
; check if CPU frequency & A/D clock need to change
	movlw		5								; ranges 5-7 do not use PLL
	subwf		Range, w						; w = range - 5
	btfsc		WREG, 7						; w is negative if range < 5
	return									; exit for Ranges 0-4
; handle ranges 5-7
	BANKSEL	OSCCON						; ADCON1 is in same bank
	bcf		OSCCON, SPLLEN				; disable PLL, Fcpu = 8MHz
	movlw		(1<<ADFM) | (b'001'<<ADCS0) | (0<<ADPREF0)
	movwf		ADCON1						; right justify data, clk=Fosc/8, VRPOS=VDD
	btfsc		STATUS, Z					; Z=1 if range = 5 (leftover from subwf)
	goto		Get.Range.100
; ranges 6-7
	movlw		(0<<SPLLEN) | (b'1100'<<IRCF0) | (b'10'<<SCS0)
	movwf		OSCCON						; disable PLL, IntOsc=2Mhz, use IntOsc
	movlw		(1<<ADFM) | (b'000'<<ADCS0) | (0<<ADPREF0)
	movwf		ADCON1						; right justify data, clk=Fosc/2, VRPOS=VDD
Get.Range.100
	BANKSEL	PORTA
	return

T1PresetTable
; values must be negative since T1 only counts up
; values for ranges 1..4 are 2 counts less than expected due to instruction
; execution time in Run.T1
; note: value for Range=0 is not used
	dw	0, 0				; 0 PLL on, Fcpu = 32MHz, T1 clock = 1MHz
	dw	LOW -.8,		 HIGH -.8			; 1
	dw	LOW -.98,	 HIGH -.98			; 2
	dw	LOW -.998,	 HIGH -.998			; 3
	dw	LOW -.9998,	 HIGH -.9998		; 4
; Fcpu = 8MHz
	dw	LOW -.25000, HIGH -.25000		; 5 PLL off, Fcpu=8MHz, T1 clock = 250KHz
; Fcpu = 2MHz
	dw	LOW -.62500, HIGH -.62500		; 6 PLL off, Fcpu=2MHz, T1 clock = 62.5KHz
	dw	LOW -.62500, HIGH -.62500		; 7
; Get.Range


Get.Range.Mode
; read the A/D
;	enter with W = A/D channel in appropriate bits and ADON bit set
;	exit with W = 3 MSbits of reading in bits 2..0
	call		AD.Read						; read the A/D
#if AD_USE_8_BITS == 0				; if using Right justified data
	rlf		ADvalue, f					; bit 7 into C
	rlf		ADvalue+1, f				; C into bit 8
	movf		ADvalue+1, w
#else										; if using left justified data
	swapf		ADvalue, w					; swap nibbles
	rrf		WREG							; bits 7..5 are now in 2..0
#endif ; AD_USE_8_BITS
	andlw		0x07							; keep only Mode bits
	return
; Get.Range.Mode


AD.Read
; read the 10 bit A/D
; enter with w = A/D channel in appropriate bits and ADON bit set
; exit with 10 bit value in ADvalue and ADvalue+1
;		and w = low byte value
; about 28 instruction cycles + about 12usec with 32MHz clock
;	total of about 16usec with 32MHz clock
	BANKSEL	ADCON0
	movwf		ADCON0						; set the channel and keep A/D on
; delay for acquisition time - 20 instruction sycles
	movlw		.7
AD.Read.05
	decfsz	WREG, f						; 1 update count, skip next if done
	goto		AD.Read.05					; 2
;
	bsf		ADCON0, ADGO				; start the conversion
AD.Read.10
	btfsc		ADCON0, ADGO				; skip next if done conversion
	goto		AD.Read.10					; wait for done

#if AD_USE_8_BITS == 0				; if using Right justified data
	movf		ADRESH, w					; get the high byte
	andlw		0x03							; clear upper 6 bits
	movwf		ADvalue+1					; save it
	movf		ADRESL, w					; get the low byte
	movwf		ADvalue						; save it
#else										; if using left justified data
	movfw		ADRESH						; get the high byte
	movwf		ADvalue						; save it
	clrf		ADvalue+1
#endif

#if AD_NO_BITS_0_1 == 1
	movlw		~3								; clear bits 0 & 1
	andwf		ADvalue, f
#endif

	BANKSEL	PORTA
	return
; AD.Read


AD.X10
; multiply  ADvalue by 10
;	enter with:
;		value in ADvalue
;	exit with: ADvalue multiplied by 10
;	uses: temp0
; save original value
	movfw		ADvalue						; get low byte
	movwf		temp0							; save it
	movfw		ADvalue+1					; get high byte
	movwf		temp0+1						; save it
; now, multiply by 10
;	*2
	lslf		ADvalue, f
	rlf		ADvalue+1, f
;	*4
	lslf		ADvalue, f
	rlf		ADvalue+1, f
;	*5
	movfw		temp0							; retrieve original low byte
	addwf		ADvalue, f					; update accumulated low byte value
	movfw		temp0+1						; retrieve original high byte value
	addwfc	ADvalue+1, f				; update accumulated high byte value
;	*10
	lslf		ADvalue, f
	rlf		ADvalue+1, f
	return
; AD.X10


;===============================================================================
;									Initialize Hardware
;===============================================================================

init	; initialize the system hardware

; Oscillator - set to 32MHz using 8MHz clock and PLL
	BANKSEL	OSCCON
	movlw		(1<<SPLLEN) | (b'1110'<<IRCF0) | (b'00'<<SCS0)
	movwf		OSCCON						; enable PLL, IntOsc=8Mhz, clock determined by CONFIG
												; *** SCS must be '00' to allow PLL operation ***
init.20
	btfss		OSCSTAT, HFIOFL			; skip next if HF osc ready
	goto		init.10
init.10
	btfss		OSCSTAT, PLLR				; skip next if PLL ready
	goto		init.10

; Port A
	BANKSEL	PORTA
	clrf		PORTA
	BANKSEL	TRISA
	movlw		TRISA_VAL
	movwf		TRISA
	BANKSEL	ANSELA
	movlw		ANSEL_VAL
	movwf		ANSELA

; A/D convertor
	BANKSEL	ADCON0
	movlw		(1<<ADON)					; A/D on, used later to select A/D channel
	movwf		ADCON0
	movlw		(1<<ADFM) | (b'010'<<ADCS0) | (0<<ADPREF0)
	movwf		ADCON1						; right justify data, clk=Fosc/32, VRPOS=VDD
	clrf		ADCON2						; use ADCON0(1<<ADGO) as trigger

; Timer 1
	BANKSEL	T1CON
	movlw		(b'00'<<TMR1CS0) | (b'11'<<T1CKPS0) | (1<<NOT_T1SYNC) ; source = FOSC/4	
	movwf		T1CON							; PRESCALE = 8:1, no sync, Timer off
	clrf		T1GCON						; do not use gate

; Interrupts
	clrf		INTCON						; clear global interrupt enable
	BANKSEL	PIE1
	clrf		PIE1							; clear peripheral interrupt enables
	BANKSEL	PORTA
	return
; init

	END
